//*******************************************************************************************************
//*******************************************************************************************************
//
//      Name:       Microvision.ino
//      Purpose:    Hardware interface for Arduino version of Microvision Emulator
//      Author:     Paul Robson
//      Date:       24th January 2014
//
//*******************************************************************************************************
//*******************************************************************************************************

#include <avr/pgmspace.h>														// Code stored in PROGMEM as we only have 2k RAM.
#include "LCD4884.h" 															// LCD Library
#include <Keypad.h>

#define ARDUINO 																// Tells code to use Arduino versions (e.g. reading code memory)
#include "general.h" 															// Includes - manually copied from the Emulator code directory
#include "core11.h"
#include "hwinterface.h"

void SPRINT(char *label,int value);

#include "/Users/paulrobson/Microvision/Emulator/core11.c" 						// Nothing else I tried would work so you'd have to manually change this !
#include "/Users/paulrobson/Microvision/Emulator/hwinterface.c"

static int displayedLCDState[16]; 												// Displayed state of LCD

const byte ROWS = 4; 															// four rows
const byte COLS = 3; 															// three columns
char keys[ROWS][COLS] = {
  {'1','2','3'},
  {'4','5','6'},
  {'7','8','9'},
  {'*','0','#'}
};
byte rowPins[ROWS] = {A5, A4, A3, A2}; 											// connect to the row pinouts of the keypad
byte colPins[COLS] = {13, 12, 11}; 												// connect to the column pinouts of the keypad

Keypad  keypad = Keypad( makeKeymap(keys), rowPins, colPins, ROWS, COLS ); 		// Create the keypad object

#define TONE_PIN	(8) 														// Use D8 to generate a beep.
#define POTENTIOMENTER_PIN (A1) 												// Pin Pot is connected to.

//*******************************************************************************************************
//											Arduino Initialisation
//*******************************************************************************************************

void setup() {
	int i;
	Serial.begin(9600); 														// Start Serial port up
  	C11_DetectRotaryControl(); 													// Detect the rotary control.
  	C11_Reset(); 																// Reset the MCU
	for (int i = 0;i < 16;i++) displayedLCDState[i] = -1; 						// Set displayed state to invalid values, force repaint of each line.
	lcd.LCD_init(); 															// Set up the LCD.
  	lcd.LCD_clear();
  	lcd.backlight(ON); 	

}

//*******************************************************************************************************
//				KeyPressed routine that actually does what its f**king name suggest.
//
//	What idiot writes a keyboard scanning routine that doesn't include a simple test to see whether a
//	key is pressed or not, and then writes a function called isPressed(key) which *doesn't* detect if a
//  key is pressed but detects a press transition. It's laughable.
//
//	Note: this accesses internal members. It's actually *not possible* to check the status of an 
//	individual key using this library. Honest. I've read the code.
//	
//*******************************************************************************************************

bool isActuallyBloodyPressed(char keyChar) {
	for (byte i=0; i<LIST_MAX; i++) {
		if ( keypad.key[i].kchar == keyChar ) {
			if ( (keypad.key[i].kstate == PRESSED))// && key[i].stateChanged ) what possible use is this ? ever ?
				return true;
		}
	}
	return false;	// Not pressed. well thanks for that, that's the obvious bit of the method.
}
//*******************************************************************************************************
//											Main loop does one frame
//*******************************************************************************************************

void loop() {
	C11_Execute(-1);
}

//*******************************************************************************************************
//										Return rotary control value 0-100
//*******************************************************************************************************

int XHWIGetRotaryControl() { 
	int lvl = analogRead(POTENTIOMENTER_PIN) / 10;
	SPRINT("POT",lvl);
	return lvl;
}

//*******************************************************************************************************
//												Read keypad state
//*******************************************************************************************************

static char *colKeys[] = { "147*","2580","369#" };

BOOL XHWIReadKeypad(int col,int row) {
	return isActuallyBloodyPressed(colKeys[col][row]);
}

//*******************************************************************************************************
//												Set Beep Frequency
//*******************************************************************************************************

void XHWISetPitch(int frequency) {
	if (frequency <= 0) noTone(TONE_PIN); else tone(TONE_PIN,frequency);
}

//*******************************************************************************************************
//
//							LCD 4884 Column Repainting code. Very specific (!)
//
//*******************************************************************************************************

static void RepaintLCDColumn(int column) {
	uint64_t bitPattern = 0; 														// Bits to be written out to given column (bit 0 = top pixel)
	int row;
	int mask = (0x8000 >> column);
	for (row = 0;row < 16;row++) { 													// Work through each row, checking if the column bit is set
		if (displayedLCDState[row] & mask) bitPattern |= (((uint64_t)7) << row*3); 	// If so, set 3 pixels on that column long at the row position.
	}
	for (row = 0;row < 6;row++) { 													// Now we work through each row of the LCD.
		lcd.LCD_set_XY(column * 4 + 10, row);										// Put at X = Column * 4 +10, Y = 0
		unsigned char b = (bitPattern & 0xFF); 										// Extract the byte
		lcd.LCD_write_byte(b,1); 													// Write it out four times.
		lcd.LCD_write_byte(b,1);
		lcd.LCD_write_byte(b,1);
		lcd.LCD_write_byte(b,1);
		bitPattern = bitPattern >> 8; 												// Shift the next byte into position to write out.
	}
}

//**********************************************************************************c*********************
//					Synchronise Display/Update Keyboard etc. End of Frame code
//*******************************************************************************************************

void XHWISynchronise(int milliseconds) {	
	int i,col,row,mask;
	int dirtyColumns = 0; 															// Bit 0-15 set if column 0-15 needs repainting.
	for (i = 0; i < 16;i++) {														// Scan through each row
		row = HWIReadLCDRow(i); 													// Get the current state.
		mask = row ^ displayedLCDState[i];											// Changed bits will be set to logic '1' now.
		dirtyColumns = dirtyColumns | mask; 										// Or into mask so we are tracking all the dirty columns.
		displayedLCDState[i] = row; 												// Update the displayed status						
	}
	for (col = 0;col < 16;col++) { 													// Work through all the columns
		if (dirtyColumns & (0x8000 >> col)) RepaintLCDColumn(col); 					// If the dirty bit is set then 
	}
	keypad.getKeys();
}

//*******************************************************************************************************
//											Simple Debugging Function
//*******************************************************************************************************

void SPRINT(char *label,int value) {
	Serial.print(label);Serial.print(':');Serial.println(value,HEX);
}

