
	protected int accumulator;													// 8021 Accumulator
	protected int programCounter;												// 8021 Program counter
	protected int stackPointer;													// 8021 Stack Pointer
	protected int timerRegister;												// 8021 Timer Register
	
	protected int[] reg;														// 8021 Register Memory
	protected int[] program;													// 8021 Program Memory
	
	protected boolean carry;													// 8021 Carry
	protected boolean auxCarry;													// 8021 Auxiliary Carry
	protected boolean timerOverflow;											// 8021 Timer Overflow
	protected boolean timerEnabled;												// 8021 Timer Enabled
	
	private int timerCount;														// The divide by 32 timer counter 
	private long totalElapsedCycles;											// Total number of elapsed cycles 								
	private I8021Hardware hardware;												// Hardware interface.
	private int breakPoint;														// Breakpoint.
	
	protected final static int ROM_SIZE = 2048;									// Memory sizes.
	protected final static int RAM_SIZE = 64;
	protected final static int TIMER_DIVIDER = 32;								// Timer divider.
	
	private final static long CLOCK_SPEED = 3580000;							// Best clock speed. (= 3.58Mhz)
	private final static int  CLOCKS_PER_CYCLE = 30;							// 30 Clocks per Cycle.
	private static final int FRAMES_PER_SECOND = 30;							// Frames per second.
	
	/**
	 * Reset the 8021/8022 Processor.
	 */
	public void reset() {
		timerCount = 0;totalElapsedCycles = 0;									// Reset our internal counters.
		programCounter = stackPointer = 0;										// Sets PC and SP to zero (page 2-25 8048 pdf)
	}
	
	/**
	 * Fetch a byte using the program counter, which is bumped.
	 * 
	 * @return	fetched byte.
	 */
	private int fetch() {
		int opcode = program[programCounter++];									// Read next instruction, bump PC
		programCounter = programCounter & 0x7FF;								// Wrap in 2k code space (8022)
		timerCount--;															// Decrement cycle/32 counter. Mostly one fetch == one cycle		
		return opcode;
	}

	/**
	 * 8 bit add to accumulator with carry and aux carry.
	 * @param n
	 * @param carryIn
	 */
	private void add(int n,boolean carryIn) {
		auxCarry = (accumulator & 0x0F)+(n & 0x0F) + (carryIn ? 1 : 0) > 15;	// Evaluate half carry from AC[3:0], Value[3:0] and carryIn. Used for AC.
		
		accumulator = accumulator + n + (carryIn ? 1 : 0);						// Work out the result
		carry = (accumulator > 0xFF);											// Carry depends on result overflow
		accumulator = accumulator & 0xFF;										// Force Accumulator in range.
	}
	
	/**
	 * Push return address on stack
	 * 
	 * @param address	Address to push
	 */
	private void push(int address) {
		int addr = stackPointer * 2 + 8;										// Stack address to use.
		stackPointer = (stackPointer + 1) & 7;									// Increment stack pointer and wrap.
		reg[addr] = address & 0xFF;												// Save LSB
		reg[addr+1] = address >> 8;												// Save MSB		
	}
	
	/**
	 * Pull return address off stack.
	 * 
	 * @return	return address
	 */
	private int pull() {
		stackPointer = (stackPointer - 1) & 7;									// Decrement stack pointer and wrap
		int addr = stackPointer * 2 + 8;										// Stack address to use.
		return (reg[addr] | (reg[addr+1] << 8)) & 0x7FF;						// Get address back and put in correct range.
	}

	/**
	 * Decimal Adjust Accumulator based on aux Carry (see 8048 4-13)
	 */
	private void decimalAdjust() {
		if (auxCarry || (accumulator & 0x0F) > 9) {								// If aux-carry set or AC[3:0] > 9 then add 6 to AC
			accumulator = accumulator + 6;
		}
		if (carry || (accumulator & 0xF0) > 0x90) {								// If carry set or AC[7:4] > 9 then add 6 to AC[7:4] and set carry.
			accumulator = accumulator + 0x60;
			carry = true;
		}
	}
	
	/**
	 * XCHD command. Like exchange but lower four nibbles only.
	 * 
	 * @param address	working address. (e.g. for xchd a,@r1 its the 5 lower bits of r1)
	 */
	private void exchangeNibble(int address) {
		int oldReg = reg[address];
		reg[address] = (reg[address] & 0xF0) | (accumulator & 0x0F);				// Calculate Rn value.
		accumulator = (accumulator & 0xF0) | (oldReg & 0x0F);						// Calculate Ac value using old Reg value.
	}
	
	/**
	 * Rotate A Left
	 */
	private void rotateLeft() {
		accumulator = accumulator << 1;
		if (accumulator > 0xFF) accumulator = (accumulator & 0xFF) | 1;
	}
	
	/**
	 * Rotate A left through carry.
	 */
	private void rotateLeftThroughCarry() {
		accumulator = (accumulator << 1) | (carry ? 1 : 0);
		carry = (accumulator > 0xFF);
		accumulator = accumulator & 0xFF;
	}

	/**
	 * 	Rotate A right
	 */
	private void rotateRight() {
		int lsb = accumulator & 1;
		accumulator = (accumulator >> 1)  | (lsb << 7);
	}
	
	/**
	 * Rotate A right through the carry
	 */
	private void rotateRightThroughCarry() {
		boolean newCarry = (accumulator & 1) != 0;
		accumulator = (accumulator >> 1) | (carry ? 0x80:0x00);
		carry = newCarry;
	}

	/**
	 * Execute a single instruction.
	 * 
	 * @return true if hit breakpoint.
	 */
	public int execute() {
		int temp;
		
		if (timerCount < 0) {														// Divide by 32 timer is out ?
			timerCount += TIMER_DIVIDER;											// Set it to it isn't any more.
			totalElapsedCycles += TIMER_DIVIDER;									// Add to total elapsed cycles
			if (timerEnabled) {														// Is the timer register enabled.
				timerRegister++;													// Bump the timer register
				if (timerRegister == 256) {											// Has it overflowed ?
					timerRegister = 0;												// Reset it to zero
					timerOverflow = true;											// And set the overflow flag.
				}
			}
		}
		switch(fetch()) {															// All code from here on automatically generated
		}
		return programCounter;
	}
	
	public int getStepOverBreak() {
		int opcode = program[programCounter];										// What we are executing next.
		if ((opcode & 0x1F) == 0x14) {												// Covers 14,34,54,74 .... F4 e.g. Call Commands.
			return programCounter+1;
		}
		return -1;
	}

}
