#!/usr/bin/env lua
------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--[[
																			TMS1100 Assembler
																			=================

																  Written by Paul Robson January 2014

	12-1-14		Updated to allow for correct order of ldx,bit operation operands. Fixed to use LFSR ordering by default. Replaced ac2ac etc. by a2aac :)
	14-1-14 	Fixed problems with page overflow reporting.
	15-1-14 	Allowed use of underscore in labels and equates.
	17-1-14 	Finally fixed page overflow reporting, wouldn't assemble into the last byte of a page !
	18-1-14 	Comments didn't work when two sets of comments on a line :)
	19-1-14 	Pseudo-operation 'include' added. Added usage.txt dump to display ROM usage.
				
--]]
------------------------------------------------------------------------------------------------------------------------------------------------------------------------

-- Standard Stub OOP create as a global.
_G.Base =  _G.Base or { new = function(s) local o = {} setmetatable(o,s) s.__index = s s:initialise() return o end, initialise = function() end }

------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--																Class which maintains a store of identifiers
------------------------------------------------------------------------------------------------------------------------------------------------------------------------

local IdentifierStore = Base:new()

function IdentifierStore:initialise()
	self.m_identifierHash = {} 																		-- hash mapping names to values
end

function IdentifierStore:get(name)																	-- retrieve identifier given a key
	return self.m_identifierHash[name:lower()]
end

function IdentifierStore:set(name,value) 															-- set identifier to a numerical value
	name = name:lower()
	assert(self.m_identifierHash[name] == nil,"Duplicate identifier "  .. name)						-- check for duplicates, not allowed
	self.m_identifierHash[name] = value 															-- store the value
	-- print(value,"->",name)
	return self 																					-- allow chaining
end

------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--																	Expression Evaluator Class
--												<base 10> $<base16> %<base2> <identifier> -<term> ><term> <<term>
------------------------------------------------------------------------------------------------------------------------------------------------------------------------

local Evaluator = Base:new()

function Evaluator:evaluate(expression, identifierStore)
	expression = expression:match("^%s*(.*)$"):lower()												-- remove leading spaces and make lower case
	while expression:match("%s$") do expression = expression:sub(1,-2) end 							-- remove trailing spaces
	local retval 																					-- return value
	if expression:match("^[%a%_][%w%_]*$") then 													-- is it an identifier ?
		retval = identifierStore:get(expression)													-- retrieve value
		assert(retval ~= nil,"Identifier " .. expression .. " is not defined")						-- check it exists
	elseif expression:match("^%d+$") then 															-- is it a decimal constant ?
		retval = tonumber(expression)
	elseif expression:match("^%$%x+$") then 														-- is it a hexadecimal constant ?
		retval = tonumber(expression:sub(2),16)																	
	elseif expression:match("^%%[01]+$") then 														-- is it a binary constant ?
		retval = tonumber(expression:sub(2),2)																	
	elseif expression:sub(1,1) == "-" then 															-- is it minus term ?
		retval = -self:evaluate(expression:sub(2),identifierStore)
	elseif expression:sub(1,1) == ">" then 															-- is it upper nibl term ?
		retval = math.floor(self:evaluate(expression:sub(2),identifierStore)/16)
	elseif expression:sub(1,1) == "<" then 															-- is it lower nibl term ?
		retval = self:evaluate(expression:sub(2),identifierStore) % 16
	else
		assert(false,"Expression syntax error") 													-- do not know what it is	
	end
	return retval
end

------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--															Simple ROM Storage Class (stores in normal order)
------------------------------------------------------------------------------------------------------------------------------------------------------------------------

local ROMMemory = Base:new()

function ROMMemory:initialise() 
	self.m_pageCount = 32 																			-- number of pages
	self.m_code = {} 																				-- code memory.
	self.m_pagePointer = {} 																		-- 32 pointers which are the current write address for each of 32 x 64 byte pages
	self.m_writeCount = {}
	for i = 0,self.m_pageCount-1 do self.m_pagePointer[i] = 0 self.m_writeCount[i] = 0 end 			-- reset those pointers.
	self.m_currentPage = 15 																		-- start on page 15 (at address $3C0)
end

function ROMMemory:write(byte) 																		-- write a byte to the current page
	assert(byte >= 0 and byte < 256)
	local address = self:getAddress() 																-- get the current writing address
	self:isWritePossible()
	self:writeDirect(address,byte) 																	-- store the byte into memory
	self.m_pagePointer[self.m_currentPage] = self:advance(self.m_pagePointer[self.m_currentPage]) 	-- bump the pointer to point to the next byte
	self.m_writeCount[self.m_currentPage] = self.m_writeCount[self.m_currentPage]+1 				-- increment write count.
	return self
end

function ROMMemory:isWritePossible() 
	assert(not self:isLastByte(self.m_pagePointer[self.m_currentPage]),"Page overflow")				-- error if at last page of code.
end

function ROMMemory:writeDirect(address,byte)														-- write byte direct to memory space.
	assert(byte >= 0 and byte < 256 and address >= 0 and address < self.m_pageCount * 64)
	--print(("%03x : %02x"):format(address,byte))													-- debug print
	self.m_code[address] = byte
end

function ROMMemory:getAddress()
	return self.m_pagePointer[self.m_currentPage] + self.m_currentPage * 64 
end

function ROMMemory:setPage(pageNumber) 																-- set the current page to write to, 0-31
	assert(pageNumber >= 0 and pageNumber < self.m_pageCount)
	self.m_currentPage = pageNumber
end

function ROMMemory:getImage() 
	local image = {}
	for i = 1,self.m_pageCount * 64 do image[i] = 0x7F end 											-- fill the image with $7F (CLA instruction)
	for address,data in pairs(self.m_code) do 														-- work through all the assembled data
		image[address+1] = data 																	-- add 1 to give an offset by 1 binary because lua indexes from 1
	end
	return image
end

function ROMMemory:isLastByte(p) return (p == 64) end												-- reached the end ?
function ROMMemory:advance(p) return p+1 end														-- advance the pointer
function ROMMemory:getPage() return self.m_currentPage end 											-- get current page

function ROMMemory:usage() 																			-- dump ROM usage to standard out.
	local total = 0
	local f = io.open("usage.txt","w")
	f:write("Code usage:\n")
	for i = 0,self.m_pageCount-1 do
		local count = self.m_writeCount[i]
		if count ~= 0 then
			f:write(("Page %2d : $%03x : %2d free\n"):format(i,i*64,64-count))
			total = total + count
		end
	end
	f:write(("Used %d bytes (%d %%)\n"):format(total,total*100/2048))
	f:close()
end

------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--																TMS1100 ROM Storage Class (stores in LFSR order)
------------------------------------------------------------------------------------------------------------------------------------------------------------------------

LFSRMemory = ROMMemory:new()

LFSRMemory.codeSequence = 																			-- code sequence in TMS1100/TMS1000
		{ 0x01, 0x03, 0x07, 0x0F, 0x1F, 0x3F, 0x3E,
		0x3D, 0x3B, 0x37, 0x2F, 0x1E, 0x3C, 0x39, 0x33,
		0x27, 0x0E, 0x1D, 0x3A, 0x35, 0x2B, 0x16, 0x2C,
		0x18, 0x30, 0x21, 0x02, 0x05, 0x0B, 0x17, 0x2E,
		0x1C, 0x38, 0x31, 0x23, 0x06, 0x0D, 0x1B, 0x36,
		0x2D, 0x1A, 0x34, 0x29, 0x12, 0x24, 0x08, 0x11,
		0x22, 0x04, 0x09, 0x13, 0x26, 0x0C, 0x19, 0x32,
		0x25, 0x0A, 0x15, 0x2A, 0x14, 0x28, 0x10, 0x20 }
LFSRMemory.codeSequence[0] = 0x00

function LFSRMemory:isLastByte(p)
	return p == nil
end

function LFSRMemory:advance(p)																		-- advance the pointer
	for i = 0,63 do 
		if LFSRMemory.codeSequence[i] == p then return LFSRMemory.codeSequence[i+1] end
	end
	assert(false,"Code problems, see source." .. p)
end

------------------------------------------------------------------------------------------------------------------------------------------------------------------------
-- 																			Assembler Class
------------------------------------------------------------------------------------------------------------------------------------------------------------------------

local Assembler = Base:new()

function Assembler:initialise() 
	self.m_reverse = { 8,4,12,2,10,6,14,1,9,5,13,3,11,7,15 } self.m_reverse[0] = 0 					-- initialise nibble reverse tables 
	self.m_reverse3 = { 4,2,6,1,5,3,7 } self.m_reverse3[0] = 0
	self.m_reverse2 = { 2,1,3} self.m_reverse2[0] = 0

	self.m_mnemonics = {																			-- mnemonic to opcode hash lookup 
		alem = 0x01, amaac = 0x06, br = 0x180, call = 0x1C0, cla = 0x7F, comc = 0x0B, comx = 0x09,	-- 0xx simple opcode
		cpaiz = 0x3D, dman = 0x07, dyn = 0x04, imac = 0x3E, iyc = 0x05,knez = 0x0E, ldp = 0x210, 	-- 1xx branch or call
		ldx = 0x328, mnea = 0x00, mnez = 0x3F, rbit = 0x434, retn = 0x0F, rstr = 0x0C, 				-- 2xx ldp
		saman = 0x3C, sbit = 0x430, setr = 0x0D, tam = 0x27, tamdyn = 0x24, tamiyc = 0x25, 			-- 3xx ldx
		tamza = 0x26, tay = 0x20, tbit1 = 0x438, tcy = 0x540, tcmiy = 0x560, tdo = 0x0A, 			-- 4xx bit op - sbit, rbit, tbit1
		tka = 0x08, tma = 0x21, tmy = 0x22, tya = 0x23, xma = 0x03, ynea = 0x02, ynec = 0x550		-- 5xx reverse nibble operand (e.g. tcy, tcmiy, ynec)
	}	
	for i = 1,15 do self.m_mnemonics["a"..i.."aac"] = self.m_reverse[i-1] + 0x70 end 				-- create a1aac to a15aac opcodes
	self.m_mnemonics.iac = self.m_mnemonics.a1aac 													-- put in code for iac and dan
	self.m_mnemonics.dan = self.m_mnemonics.a15aac

	self.m_evaluator = Evaluator:new() 																-- evaluator for expressions
	self.m_identifierStore = IdentifierStore:new() 													-- store for identifiers
	self.m_memory = LFSRMemory:new() 																-- store for program code
	self.m_references = {} 																			-- forward references (BR, CALL and LDP)
	self.m_lineNumber = 0 																			-- current line number
	self.m_fileName = "?"
end

function Assembler:setFileName(n) self.m_fileName = n end
function Assembler:getFileName() return self.m_fileName end

function Assembler:assembleLine(line) 																-- assemble a single line
	self.m_lineNumber = self.m_lineNumber + 1 														-- go to the next line.
	return self:_assembleLine(line)																	-- call the non-incrementing line assembler
end

function Assembler:_assembleLine(line)																-- assemble a line, don't change the line number.

	while line:match("//") do line = line:match("^(.*)//") end 										-- remove comments
	line = line:lower() 																			-- make it lower case
	while line:match("%s$") do line = line:sub(1,-2) end 											-- remove trailing spaces.

	local label, rest = line:match("^([%w%_]+)%s*(.*)") 											-- is it a label of some sort ?
	if label ~= nil then 																		
		if rest:sub(1,1) == "=" then 																-- is it an equate
			local v = self.m_evaluator:evaluate(rest:sub(2),self.m_identifierStore)					-- evaluate the equate and store it.
			self.m_identifierStore:set(label,v)
			line = "" 																				-- nothing else on this line.
		else
			self.m_identifierStore:set(label,self.m_memory:getAddress())							-- set label equal to current address
			line = rest 																			-- and process the rest of the line.
		end
	end
	if line:match("^%s*$") then return self end 													-- crash out if nothing left on this line

	local opcode,operand = line:match("^%s*(%w+)%s*(.*)$")											-- split remainder of instruction into opcode and possible operand
	assert(opcode ~= nil,"Syntax error")															-- did it work

	if opcode == "include" then 																	-- include a file.
		local oldLineNumber = self.m_lineNumber 													-- save line number
		local oldName = self.m_fileName
		self.m_lineNumber = 0 																		-- reset line number
		local inc = io.open(operand,"r") 															-- open include file
		self.m_fileName = operand
		for l in inc:lines() do self:assembleLine(l) end 											-- assemble it
		inc:close() 																				-- close it.
		self.m_lineNumber = oldLineNumber 															-- restore line number.
		self.m_fileName = oldName
		return self
	end

	if opcode == "lcall" or opcode == "lbr" then 													-- handle lcall, lbr 
		self:_assembleLine(" ldp "..operand) 														-- assemble ldp x
		self:_assembleLine(" " .. opcode:sub(2).." "..operand)												-- assemble call x or br x ß
		return self
	end

	if opcode == "page" then 																		-- if page
		if operand == "" then
			local page = self.m_memory:getPage() 													-- current page (inc. chapter)
			local chapter = math.floor(page / 16) 													-- current chapter
			page = (page + 1) % 16 + chapter * 16 													-- next page but keep the same chapter
			self.m_memory:setPage(page) 															-- and go there.
		else
			local newPage = self.m_evaluator:evaluate(operand,self.m_identifierStore)				-- work out the new page number
			self.m_memory:setPage(newPage)															-- switch to that page
		end
		return self																					-- and exit immediately.
	end
	local byte = self.m_mnemonics[opcode] 															-- look up the opcode in the table
	assert(byte ~= nil,"Opcode "..opcode.." is unknown.")											-- check it is a valid opcode.

	self.m_memory:isWritePossible() 																-- can we write here (page overflow ?)

	if byte >= 0x100 then 																			-- is it not a simple one word operand ?
		local asmType = math.floor(byte / 256) 														-- identify what type it is.
		if asmType ~= 1 and asmType ~= 2 then 														-- if it is not BR, CALL or LDP (which are forward references)
			operand = self.m_evaluator:evaluate(operand,self.m_identifierStore)						-- evaluate the operand (BR, CALL and LDP may not allow this)
		end
		if asmType == 1 or asmType == 2 then 														-- BR/CALL and LDP type
			byte = byte % 256 																		-- throw away the type identifier - use the code as a base.
			local newReference = { address = self.m_memory:getAddress(), opcode = byte,				-- create a reference 
																				operand = operand }
			self.m_references[#self.m_references+1] = newReference 									-- add to the reference list.
		elseif asmType == 3 then 																	-- LDX type
			assert(operand >= 0 and operand <= 7,"ldx operand must be 0-7")
			byte = byte % 256 + self.m_reverse3[operand]
		elseif asmType == 4 then 																	-- SBIT, RBIT, TBIT1 type
			assert(operand >= 0 and operand <= 3,"bit operation operand must be 0-3")
			byte = byte % 256 + self.m_reverse2[operand]
		elseif asmType == 5 then  																	-- 4 bit operand reversed (TCMIY, TCY, YNEC)
			assert(operand >= 0 and operand <= 15,"operand must be 0-15")
			byte = byte % 256 + self.m_reverse[operand]
		else
			assert(false,"type not decoded "..asmType)
		end
	end
	self.m_memory:write(byte) 																		-- write the value out.
	return self
end

function Assembler:fixReferences() 																	-- fix all references (e.g. LDP, BR and CALL)
	self.m_lineNumber = 0 																			-- no current line number
	for _,ref in pairs(self.m_references) do 														-- work through all known references.
		local address = self.m_evaluator:evaluate(ref.operand,self.m_identifierStore)				-- identify the final address by evaluating the operand
		if ref.opcode == 0x10 then
			address = math.floor(address / 64) % 16 												-- if LDP (0x1x) then it is a page number 0-15
			address = self.m_reverse[address] 														-- and it is stored in the reverse format
		else
			address = address % 64 																	-- otherwise it is BR or CALL and an address in a page
		end
		self.m_memory:writeDirect(ref.address,ref.opcode + address)									-- patch it
	end
	self.m_references = {} 																			-- clear all the references
end

function Assembler:getBinary() 																		-- access the final binary
	return self.m_memory:getImage()
end

function Assembler:getLineNumber()
	return self.m_lineNumber
end

------------------------------------------------------------------------------------------------------------------------------------------------------------------------
-- 																Class that assembles one file into another
------------------------------------------------------------------------------------------------------------------------------------------------------------------------

AssemblerProcessor = Base:new()

function AssemblerProcessor:_assemble(source,binary) 												-- assemble source into binary
	self.m_worker = Assembler:new()																	-- create new assembler object
	self.m_worker:setFileName(source)
	local src = io.open(source,"r")
	assert(src ~= nil,"Can't open source file "..source)
	for l in src:lines() do self.m_worker:assembleLine(l) end 										-- assemble the source file
	self.m_worker:fixReferences() 																	-- fix up the references 
	src:close()																						-- close the source file
	local img = self.m_worker:getBinary() 															-- get the binary image
	if binary == nil then 																			-- no image file given
		binary = source:match("^(.*)%.asm$") or "code"												-- get the file part of the filename or use code if stuck.
		binary = binary .. ".bin"																	-- append .bin
	end
	local obj = io.open(binary,"wb") 																-- open the binary file
	obj:write(string.char(unpack(img)))																-- write the binary image out
	obj:close()																						-- close the file
	self.m_worker.m_memory:usage()
	return nil 																						-- and successful.
end

function AssemblerProcessor:assemble(source,binary) 
	local result,msg = pcall(function() self:_assemble(source,binary) end)							-- call assemble wrapped in an error manager
	if result then return nil end 																	-- if successful, return nil.
	return msg:match("%:%d+%:%s*(.*)$") .. " (" .. self.m_worker:getLineNumber() .. " : " ..		-- return processed error message
																		self.m_worker:getFileName() .. ")"
end

if #arg == 0 then 																					-- no parameters
	print("TMS1100 Assembler in lua : invoke with lua "..arg[0].." <source> [<binary>]")			-- display information
	print("Written by Paul Robson January 2014")
else
	r = AssemblerProcessor:new():assemble(arg[1],arg[2])											-- process the file names given
	if r ~= nil then print(r) end																	-- output message if an error occurs.
end
